(function(){
  const data = window.BANNER_DATA;

  const cupSelect = document.getElementById("cupSelect");
  const teamLeftSelect = document.getElementById("teamLeftSelect");
  const teamRightSelect = document.getElementById("teamRightSelect");
  const leftTopSelect = document.getElementById("leftTopSelect");
  const leftBottomSelect = document.getElementById("leftBottomSelect");
  const rightTopSelect = document.getElementById("rightTopSelect");
  const rightBottomSelect = document.getElementById("rightBottomSelect");
  const swapLeft = document.getElementById("swapLeft");
  const swapRight = document.getElementById("swapRight");
  const exportBtn = document.getElementById("exportPng");

  const cupLogo = document.querySelector(".cupmark");
  const trophyImg = document.querySelector(".trophy");
  const leftBigImg = document.getElementById("leftBigImg");
  const rightBigImg = document.getElementById("rightBigImg");
  const leftSmallImg = document.getElementById("leftSmallImg");
  const rightSmallImg = document.getElementById("rightSmallImg");
  const teamLeftText = document.getElementById("teamLeftText");
  const vsLine = document.getElementById("vsLine");
  const teamsBox = document.getElementById("teamsBox");

  const PLACEHOLDER_IMG = "data:image/svg+xml;utf8," + encodeURIComponent(
    '<svg xmlns="http://www.w3.org/2000/svg" width="600" height="900" viewBox="0 0 600 900">' +
    '<rect width="600" height="900" fill="#0b1220"/>' +
    '<circle cx="300" cy="220" r="90" fill="#1f2937" />' +
    '<rect x="150" y="340" width="300" height="360" rx="60" fill="#1f2937" />' +
    '<text x="300" y="780" text-anchor="middle" font-family="Arial" font-size="36" fill="#94a3b8">JUGADOR</text>' +
    '</svg>'
  );

  const state = {
    cupId: data.cups[0].id,
    leftTeamId: "barcelona",
    rightTeamId: "real-madrid",
    leftTopId: "lamine-yamal",
    leftBottomId: "pedri",
    rightTopId: "vinicius-junior",
    rightBottomId: "kylian-mbappe"
  };

  let scaleRaf = null;
  function updateScale(){
    const stage = document.querySelector(".stage");
    const wrapShell = document.querySelector(".wrap-shell");
    if (!stage || !wrapShell) return;
    const bannerSize = 1080;
    const bounds = stage.getBoundingClientRect();
    const availableW = bounds.width;
    const availableH = window.innerHeight - bounds.top - 24;
    const scale = Math.min(1, availableW / bannerSize, availableH / bannerSize);
    wrapShell.style.setProperty("--scale", scale.toFixed(4));
  }

  function scheduleUpdateScale(){
    if (scaleRaf) cancelAnimationFrame(scaleRaf);
    scaleRaf = requestAnimationFrame(() => {
      updateScale();
      scaleRaf = null;
    });
  }

  function getTeamById(id){
    return data.teams.find(t => t.id === id);
  }

  function getCupById(id){
    return data.cups.find(c => c.id === id);
  }

  function getAllowedTeams(cup){
    if (!cup || cup.teams === "*") return data.teams.map(t => t.id);
    return cup.teams.slice();
  }

  function setSelectOptions(select, options, selectedId){
    select.innerHTML = "";
    options.forEach(opt => {
      const el = document.createElement("option");
      el.value = opt.id;
      el.textContent = opt.name;
      if (opt.id === selectedId) el.selected = true;
      select.appendChild(el);
    });
  }

  function setPlayerOptions(select, players, selectedId){
    select.innerHTML = "";
    if (!players.length){
      const opt = document.createElement("option");
      opt.value = "";
      opt.textContent = "Sin jugadores";
      select.appendChild(opt);
      select.disabled = true;
      return;
    }
    select.disabled = false;
    players.forEach(p => {
      const el = document.createElement("option");
      el.value = p.id;
      el.textContent = p.name;
      if (p.id === selectedId) el.selected = true;
      select.appendChild(el);
    });
  }

  function getPlayer(team, playerId){
    return team.players.find(p => p.id === playerId);
  }

  function pickDefaultPlayers(team){
    if (!team.players.length) return { top: "", bottom: "" };
    const featured = (team.featured || []).filter(id => getPlayer(team, id));
    if (featured.length >= 2){
      return { top: featured[0], bottom: featured[1] };
    }
    if (team.players.length === 1){
      return { top: team.players[0].id, bottom: team.players[0].id };
    }
    return { top: team.players[0].id, bottom: team.players[1].id };
  }

  function resolveImage(player, side){
    if (!player) return PLACEHOLDER_IMG;
    const preferred = player.images ? player.images[side] : "";
    if (preferred) return preferred;
    const fallback = side === "left" ? player.images?.right : player.images?.left;
    return fallback || PLACEHOLDER_IMG;
  }

  function updateCup(){
    const cup = getCupById(state.cupId);
    if (!cup) return;
    cupLogo.src = cup.logo;
    cupLogo.alt = "Logo de " + cup.name;
    trophyImg.src = cup.trophy;
    trophyImg.alt = "Trofeo de " + cup.name;

    const allowedIds = getAllowedTeams(cup);
    const allowedTeams = data.teams.filter(t => allowedIds.includes(t.id));

    if (!allowedIds.includes(state.leftTeamId)){
      state.leftTeamId = allowedTeams[0]?.id || data.teams[0].id;
    }
    if (!allowedIds.includes(state.rightTeamId)){
      state.rightTeamId = allowedTeams[1]?.id || allowedTeams[0]?.id || data.teams[0].id;
    }

    setSelectOptions(teamLeftSelect, allowedTeams, state.leftTeamId);
    setSelectOptions(teamRightSelect, allowedTeams, state.rightTeamId);

    updateTeams();
  }

  function updateTeams(){
    const leftTeam = getTeamById(state.leftTeamId);
    const rightTeam = getTeamById(state.rightTeamId);

    if (leftTeam){
      if (!getPlayer(leftTeam, state.leftTopId) || !getPlayer(leftTeam, state.leftBottomId)){
        const defaults = pickDefaultPlayers(leftTeam);
        state.leftTopId = defaults.top;
        state.leftBottomId = defaults.bottom;
      }
      setPlayerOptions(leftTopSelect, leftTeam.players, state.leftTopId);
      setPlayerOptions(leftBottomSelect, leftTeam.players, state.leftBottomId);
    }

    if (rightTeam){
      if (!getPlayer(rightTeam, state.rightTopId) || !getPlayer(rightTeam, state.rightBottomId)){
        const defaults = pickDefaultPlayers(rightTeam);
        state.rightTopId = defaults.top;
        state.rightBottomId = defaults.bottom;
      }
      setPlayerOptions(rightTopSelect, rightTeam.players, state.rightTopId);
      setPlayerOptions(rightBottomSelect, rightTeam.players, state.rightBottomId);
    }

    updateBanner();
  }

  function updateBanner(){
    const leftTeam = getTeamById(state.leftTeamId);
    const rightTeam = getTeamById(state.rightTeamId);

    const leftTop = leftTeam ? getPlayer(leftTeam, state.leftTopId) : null;
    const leftBottom = leftTeam ? getPlayer(leftTeam, state.leftBottomId) : null;
    const rightTop = rightTeam ? getPlayer(rightTeam, state.rightTopId) : null;
    const rightBottom = rightTeam ? getPlayer(rightTeam, state.rightBottomId) : null;

    leftBigImg.src = resolveImage(leftTop, "left");
    leftBigImg.alt = leftTop ? leftTop.name : "";
    leftSmallImg.src = resolveImage(leftBottom, "left");
    leftSmallImg.alt = leftBottom ? leftBottom.name : "";

    rightBigImg.src = resolveImage(rightTop, "right");
    rightBigImg.alt = rightTop ? rightTop.name : "";
    rightSmallImg.src = resolveImage(rightBottom, "right");
    rightSmallImg.alt = rightBottom ? rightBottom.name : "";

    const leftName = leftTeam ? leftTeam.name : "";
    const rightName = rightTeam ? rightTeam.name : "";

    teamLeftText.textContent = leftName;
    vsLine.textContent = "VS " + rightName;

    const wrap = document.querySelector(".wrap");
    if (wrap){
      wrap.setAttribute("aria-label", "Banner " + leftName + " vs " + rightName);
    }

    scheduleFitTeamText();
  }

  function readCssVar(style, name, fallback){
    const raw = style.getPropertyValue(name).trim();
    return raw || fallback;
  }

  function readNumberVar(style, name, fallback){
    const raw = readCssVar(style, name, "");
    const num = parseFloat(raw);
    return Number.isFinite(num) ? num : fallback;
  }

  function readPercentVar(style, name, fallback){
    const raw = readCssVar(style, name, "");
    if (!raw) return fallback;
    if (raw.endsWith("%")) return parseFloat(raw) / 100;
    if (raw.endsWith("px")){
      const bannerSize = readNumberVar(style, "--banner-size", 1080);
      return bannerSize ? parseFloat(raw) / bannerSize : fallback;
    }
    const num = parseFloat(raw);
    return Number.isFinite(num) ? num / 100 : fallback;
  }

  function getLayoutVars(){
    const style = getComputedStyle(document.documentElement);
    const bannerSize = readNumberVar(style, "--banner-size", 1080);
    const pct = name => readPercentVar(style, name, 0);
    return {
      bannerSize,
      topLogoY: bannerSize * pct("--top-logo-y"),
      topLogoW: bannerSize * pct("--top-logo-w"),
      cupLogoY: bannerSize * pct("--cup-logo-y"),
      cupLogoW: bannerSize * pct("--cup-logo-w"),
      trophyY: bannerSize * pct("--trophy-y"),
      trophyW: bannerSize * pct("--trophy-w"),
      trophyOpacity: readNumberVar(style, "--trophy-opacity", 0.85),
      trophyFadeStart: readPercentVar(style, "--trophy-fade-start", 0.62),
      trophyFadeEnd: readPercentVar(style, "--trophy-fade-end", 0.92),
      playerBigY: bannerSize * pct("--player-big-y"),
      playerBigW: bannerSize * pct("--player-big-w"),
      bigGap: bannerSize * pct("--big-gap"),
      playerSmallY: bannerSize * pct("--player-small-y"),
      playerSmallW: bannerSize * pct("--player-small-w"),
      fadeStart: readPercentVar(style, "--fade-start", 0.68),
      fadeEnd: readPercentVar(style, "--fade-end", 0.9),
      centerBarW: bannerSize * pct("--center-bar-w"),
      centerBarY: bannerSize * pct("--center-bar-y"),
      centerBarH: bannerSize * pct("--center-bar-h"),
      centerTextPad: readNumberVar(style, "--center-text-pad", 32),
      titleFont: readCssVar(style, "--title-font", '"Bebas Neue", sans-serif'),
      uiFont: readCssVar(style, "--ui-font", "system-ui, sans-serif")
    };
  }

  function loadImage(src){
    return new Promise(resolve => {
      if (!src){
        resolve(null);
        return;
      }
      const img = new Image();
      img.onload = () => resolve(img);
      img.onerror = () => resolve(null);
      img.src = src;
    });
  }

  function drawCover(ctx, img, x, y, w, h, scale = 1){
    const sw = img.naturalWidth || 1;
    const sh = img.naturalHeight || 1;
    const targetW = w * scale;
    const targetH = h * scale;
    const ratio = Math.max(targetW / sw, targetH / sh);
    const dw = sw * ratio;
    const dh = sh * ratio;
    const dx = x + (w - dw) / 2;
    const dy = y + (h - dh) / 2;
    ctx.drawImage(img, dx, dy, dw, dh);
  }

  function createMaskedCanvas(img, width, fadeStart, fadeEnd){
    const w = Math.max(1, Math.round(width));
    const h = Math.max(1, Math.round(width * ((img.naturalHeight || 1) / (img.naturalWidth || 1))));
    const c = document.createElement("canvas");
    c.width = w;
    c.height = h;
    const cctx = c.getContext("2d");
    cctx.drawImage(img, 0, 0, w, h);
    const g = cctx.createLinearGradient(0, 0, 0, h);
    g.addColorStop(0, "rgba(0,0,0,1)");
    g.addColorStop(fadeStart, "rgba(0,0,0,1)");
    g.addColorStop(fadeEnd, "rgba(0,0,0,0)");
    cctx.globalCompositeOperation = "destination-in";
    cctx.fillStyle = g;
    cctx.fillRect(0, 0, w, h);
    cctx.globalCompositeOperation = "source-over";
    return c;
  }

  function drawMaskedImage(ctx, img, x, y, width, fadeStart, fadeEnd){
    const masked = createMaskedCanvas(img, width, fadeStart, fadeEnd);
    ctx.drawImage(masked, x, y);
    return masked.height;
  }

  function roundRect(ctx, x, y, w, h, r){
    const radius = Math.min(r, w / 2, h / 2);
    ctx.beginPath();
    ctx.moveTo(x + radius, y);
    ctx.arcTo(x + w, y, x + w, y + h, radius);
    ctx.arcTo(x + w, y + h, x, y + h, radius);
    ctx.arcTo(x, y + h, x, y, radius);
    ctx.arcTo(x, y, x + w, y, radius);
    ctx.closePath();
  }

  function drawAtmo(ctx, size){
    const stops = [
      { x: 0.25, y: 0.4, alpha: 0.22, stop: 0.6 },
      { x: 0.75, y: 0.35, alpha: 0.25, stop: 0.62 },
      { x: 0.5, y: 1.0, alpha: 0.18, stop: 0.55 },
      { x: 0.5, y: 0.1, alpha: 0.12, stop: 0.5 }
    ];
    ctx.save();
    ctx.globalCompositeOperation = "screen";
    ctx.globalAlpha = 0.7;
    stops.forEach(s => {
      const cx = size * s.x;
      const cy = size * s.y;
      const r = Math.min(cx, cy, size - cx, size - cy);
      const g = ctx.createRadialGradient(cx, cy, 0, cx, cy, r);
      g.addColorStop(0, `rgba(255,255,255,${s.alpha})`);
      g.addColorStop(s.stop, "rgba(255,255,255,0)");
      ctx.fillStyle = g;
      ctx.fillRect(0, 0, size, size);
    });
    ctx.restore();
  }

  function drawNoise(ctx, size){
    const noise = document.createElement("canvas");
    const nSize = 220;
    noise.width = nSize;
    noise.height = nSize;
    const nctx = noise.getContext("2d");
    for (let i = 0; i < 420; i++){
      const x = Math.random() * nSize;
      const y = Math.random() * nSize;
      const a = 0.1 + Math.random() * 0.2;
      nctx.fillStyle = `rgba(255,255,255,${a})`;
      nctx.fillRect(x, y, 1, 1);
    }
    ctx.save();
    ctx.globalAlpha = 0.12;
    ctx.globalCompositeOperation = "overlay";
    ctx.fillStyle = ctx.createPattern(noise, "repeat");
    ctx.fillRect(0, 0, size, size);
    ctx.restore();
  }

  function fitFontSize(ctx, text, baseSize, minSize, maxWidth, fontFamily, weight){
    let size = baseSize;
    ctx.font = `${weight} ${size}px ${fontFamily}`;
    if (ctx.measureText(text).width <= maxWidth) return size;
    const scale = maxWidth / ctx.measureText(text).width;
    size = Math.max(minSize, Math.floor(size * scale));
    return size;
  }

  function drawCenterBar(ctx, vars){
    const x = (vars.bannerSize - vars.centerBarW) / 2;
    const y = vars.centerBarY;
    const w = vars.centerBarW;
    const h = vars.centerBarH;
    const g = ctx.createLinearGradient(0, y, 0, y + h);
    g.addColorStop(0, "rgba(0,0,0,0)");
    g.addColorStop(0.18, "rgba(0,0,0,.52)");
    g.addColorStop(0.4, "rgba(0,0,0,.72)");
    g.addColorStop(0.6, "rgba(0,0,0,.78)");
    g.addColorStop(0.82, "rgba(0,0,0,.58)");
    g.addColorStop(1, "rgba(0,0,0,0)");
    ctx.save();
    ctx.shadowColor = "rgba(0,0,0,.55)";
    ctx.shadowBlur = 60;
    ctx.shadowOffsetY = 24;
    roundRect(ctx, x, y, w, h, 10);
    ctx.fillStyle = g;
    ctx.fill();
    ctx.restore();
    return { x, y, w, h };
  }

  function drawCenterText(ctx, vars, barRect){
    const leftText = teamLeftText.textContent || "";
    const vsText = vsLine.textContent || "";
    const dateText = document.getElementById("matchDate")?.textContent || "";
    const maxWidth = barRect.w - vars.centerTextPad * 2;
    const minSize = 20;
    const teamSize = fitFontSize(ctx, leftText, 60, minSize, maxWidth, vars.titleFont, 600);
    const vsSize = fitFontSize(ctx, vsText, 52, minSize, maxWidth, vars.titleFont, 600);
    const dateSize = 18;
    const gap = 12;
    const teamHeight = teamSize * 0.95;
    const vsHeight = vsSize * 0.95;
    const dateHeight = dateSize * 1.1;
    const total = teamHeight + vsHeight + dateHeight + gap * 2;
    let y = barRect.y + (barRect.h - total) / 2;
    const centerX = vars.bannerSize / 2;

    ctx.save();
    ctx.textAlign = "center";
    ctx.textBaseline = "top";
    ctx.shadowColor = "rgba(0,0,0,.7)";
    ctx.shadowBlur = 26;
    ctx.shadowOffsetY = 10;
    ctx.fillStyle = "#ffffff";
    ctx.font = `600 ${teamSize}px ${vars.titleFont}`;
    ctx.fillText(leftText, centerX, y);
    y += teamHeight + gap;
    ctx.font = `600 ${vsSize}px ${vars.titleFont}`;
    ctx.fillText(vsText, centerX, y);
    y += vsHeight + gap;
    ctx.shadowColor = "rgba(0,0,0,.75)";
    ctx.shadowBlur = 18;
    ctx.shadowOffsetY = 8;
    ctx.fillStyle = "rgba(255,255,255,0.9)";
    ctx.font = `500 ${dateSize}px ${vars.uiFont}`;
    ctx.fillText(dateText, centerX, y);
    ctx.restore();
  }

  async function renderBannerToCanvas(){
    const vars = getLayoutVars();
    const size = vars.bannerSize;
    const canvas = document.createElement("canvas");
    canvas.width = size;
    canvas.height = size;
    const ctx = canvas.getContext("2d");

    ctx.fillStyle = "#111111";
    ctx.fillRect(0, 0, size, size);

    const bgSrc = document.querySelector(".bg img")?.src;
    const brandSrc = document.querySelector(".brand")?.src;
    const cupSrc = cupLogo?.src;
    const trophySrc = trophyImg?.src;
    const leftBigSrc = leftBigImg?.src;
    const rightBigSrc = rightBigImg?.src;
    const leftSmallSrc = leftSmallImg?.src;
    const rightSmallSrc = rightSmallImg?.src;

    const [bg, brand, cup, trophy, leftBig, rightBig, leftSmall, rightSmall] = await Promise.all([
      loadImage(bgSrc),
      loadImage(brandSrc),
      loadImage(cupSrc),
      loadImage(trophySrc),
      loadImage(leftBigSrc),
      loadImage(rightBigSrc),
      loadImage(leftSmallSrc),
      loadImage(rightSmallSrc)
    ]);

    if (bg){
      drawCover(ctx, bg, 0, 0, size, size, 1.02);
    }

    drawAtmo(ctx, size);

    if (brand){
      const w = vars.topLogoW;
      const h = w * ((brand.naturalHeight || 1) / (brand.naturalWidth || 1));
      const x = (size - w) / 2;
      const y = vars.topLogoY;
      ctx.save();
      ctx.shadowColor = "rgba(0,0,0,.45)";
      ctx.shadowBlur = 18;
      ctx.shadowOffsetY = 8;
      ctx.drawImage(brand, x, y, w, h);
      ctx.restore();
    }

    if (cup){
      const w = vars.cupLogoW;
      const h = w * ((cup.naturalHeight || 1) / (cup.naturalWidth || 1));
      const x = (size - w) / 2;
      const y = vars.cupLogoY;
      ctx.save();
      ctx.shadowColor = "rgba(0,0,0,.55)";
      ctx.shadowBlur = 18;
      ctx.shadowOffsetY = 10;
      ctx.drawImage(cup, x, y, w, h);
      ctx.restore();
    }

    if (trophy){
      const w = vars.trophyW;
      const x = (size - w) / 2;
      const y = vars.trophyY;
      ctx.save();
      ctx.globalAlpha = vars.trophyOpacity;
      ctx.shadowColor = "rgba(0,0,0,.55)";
      ctx.shadowBlur = 50;
      ctx.shadowOffsetY = 24;
      drawMaskedImage(ctx, trophy, x, y, w, vars.trophyFadeStart, vars.trophyFadeEnd);
      ctx.restore();
    }

    const centerX = size / 2;
    const leftBigX = centerX - vars.playerBigW - vars.bigGap / 2;
    const rightBigX = centerX + vars.bigGap / 2;

    if (leftBig){
      ctx.save();
      ctx.shadowColor = "rgba(0,0,0,.55)";
      ctx.shadowBlur = 28;
      ctx.shadowOffsetY = 22;
      drawMaskedImage(ctx, leftBig, leftBigX, vars.playerBigY, vars.playerBigW, vars.fadeStart, vars.fadeEnd);
      ctx.restore();
    }

    if (rightBig){
      ctx.save();
      ctx.shadowColor = "rgba(0,0,0,.55)";
      ctx.shadowBlur = 28;
      ctx.shadowOffsetY = 22;
      drawMaskedImage(ctx, rightBig, rightBigX, vars.playerBigY, vars.playerBigW, vars.fadeStart, vars.fadeEnd);
      ctx.restore();
    }

    const leftSmallX = size * 0.18;
    const rightSmallX = size - size * 0.18 - vars.playerSmallW;

    if (leftSmall){
      ctx.save();
      ctx.shadowColor = "rgba(0,0,0,.55)";
      ctx.shadowBlur = 28;
      ctx.shadowOffsetY = 22;
      drawMaskedImage(ctx, leftSmall, leftSmallX, vars.playerSmallY, vars.playerSmallW, vars.fadeStart, vars.fadeEnd);
      ctx.restore();
    }

    if (rightSmall){
      ctx.save();
      ctx.shadowColor = "rgba(0,0,0,.55)";
      ctx.shadowBlur = 28;
      ctx.shadowOffsetY = 22;
      drawMaskedImage(ctx, rightSmall, rightSmallX, vars.playerSmallY, vars.playerSmallW, vars.fadeStart, vars.fadeEnd);
      ctx.restore();
    }

    const barRect = drawCenterBar(ctx, vars);
    drawCenterText(ctx, vars, barRect);
    drawNoise(ctx, size);

    return canvas;
  }

  let fitRaf = null;
  function scheduleFitTeamText(){
    if (fitRaf) cancelAnimationFrame(fitRaf);
    fitRaf = requestAnimationFrame(() => {
      fitTeamText();
      fitRaf = null;
    });
  }

  function fitTextToWidth(el, maxWidth, minSize){
    if (!el || !maxWidth) return;
    el.style.fontSize = "";
    // Force reflow so measurements use the reset size
    void el.offsetWidth;
    const baseSize = parseFloat(getComputedStyle(el).fontSize);
    if (el.scrollWidth <= maxWidth) return;
    const scale = Math.min(1, maxWidth / el.scrollWidth);
    const nextSize = Math.max(minSize, baseSize * scale);
    el.style.fontSize = nextSize + "px";
  }

  function fitTeamText(){
    const minSize = 20;
    const centerBar = document.querySelector(".center-bar");
    if (!centerBar) return;
    const style = getComputedStyle(centerBar);
    const paddingLeft = parseFloat(style.paddingLeft) || 0;
    const paddingRight = parseFloat(style.paddingRight) || 0;
    const maxWidth = Math.max(0, centerBar.clientWidth - paddingLeft - paddingRight);
    fitTextToWidth(teamLeftText, maxWidth, minSize);
    fitTextToWidth(vsLine, maxWidth, minSize);
  }

  cupSelect.addEventListener("change", () => {
    state.cupId = cupSelect.value;
    updateCup();
  });

  teamLeftSelect.addEventListener("change", () => {
    state.leftTeamId = teamLeftSelect.value;
    updateTeams();
  });

  teamRightSelect.addEventListener("change", () => {
    state.rightTeamId = teamRightSelect.value;
    updateTeams();
  });

  leftTopSelect.addEventListener("change", () => {
    state.leftTopId = leftTopSelect.value;
    updateBanner();
  });

  leftBottomSelect.addEventListener("change", () => {
    state.leftBottomId = leftBottomSelect.value;
    updateBanner();
  });

  rightTopSelect.addEventListener("change", () => {
    state.rightTopId = rightTopSelect.value;
    updateBanner();
  });

  rightBottomSelect.addEventListener("change", () => {
    state.rightBottomId = rightBottomSelect.value;
    updateBanner();
  });

  swapLeft.addEventListener("click", () => {
    const temp = state.leftTopId;
    state.leftTopId = state.leftBottomId;
    state.leftBottomId = temp;
    updateTeams();
  });

  swapRight.addEventListener("click", () => {
    const temp = state.rightTopId;
    state.rightTopId = state.rightBottomId;
    state.rightBottomId = temp;
    updateTeams();
  });

  window.addEventListener("resize", scheduleFitTeamText);
  window.addEventListener("resize", scheduleUpdateScale);

  function init(){
    setSelectOptions(cupSelect, data.cups, state.cupId);
    updateCup();
    scheduleUpdateScale();
    scheduleFitTeamText();
  }

  init();

  if (document.fonts && document.fonts.ready){
    document.fonts.ready.then(() => {
      scheduleUpdateScale();
      scheduleFitTeamText();
    });
  }

  const stageEl = document.querySelector(".stage");
  if (stageEl && "ResizeObserver" in window){
    const ro = new ResizeObserver(scheduleUpdateScale);
    ro.observe(stageEl);
  }

  if (exportBtn){
    exportBtn.addEventListener("click", async () => {
      exportBtn.disabled = true;
      exportBtn.textContent = "Exportando...";

      try{
        if (document.fonts && document.fonts.ready){
          await document.fonts.ready;
        }
        const canvas = await renderBannerToCanvas();
        const link = document.createElement("a");
        link.download = "banner.png";
        link.href = canvas.toDataURL("image/png");
        link.click();
      }catch(err){
        console.error(err);
        alert("No se pudo exportar el PNG. Intenta de nuevo.");
      }finally{
        exportBtn.disabled = false;
        exportBtn.textContent = "Exportar PNG";
      }
    });
  }
})();
