<?php

require __DIR__ . '/auth.php';
$user = require_login_json();

header('Content-Type: application/json; charset=utf-8');

if (($_SERVER['REQUEST_METHOD'] ?? 'GET') === 'GET') {
    echo json_encode(['ok' => true, 'data' => load_proposals()]);
    exit;
}

if (($_SERVER['REQUEST_METHOD'] ?? 'GET') !== 'POST') {
    http_response_code(405);
    echo json_encode(['ok' => false, 'error' => 'method_not_allowed']);
    exit;
}

$payload = json_decode(file_get_contents('php://input'), true);
if (!is_array($payload)) {
    http_response_code(400);
    echo json_encode(['ok' => false, 'error' => 'invalid_json']);
    exit;
}

$action = (string) ($payload['action'] ?? '');

try {
    $pdo = require __DIR__ . '/db.php';
} catch (Throwable $error) {
    http_response_code(500);
    echo json_encode(['ok' => false, 'error' => 'db_connection']);
    exit;
}

if ($action === 'create_post') {
    if ($user['role'] !== 'creator') {
        http_response_code(403);
        echo json_encode(['ok' => false, 'error' => 'forbidden']);
        exit;
    }

    $title = trim((string) ($payload['title'] ?? ''));
    $slides = $payload['slides'] ?? [];

    if ($title === '' || !is_array($slides) || count($slides) === 0) {
        http_response_code(400);
        echo json_encode(['ok' => false, 'error' => 'missing_fields']);
        exit;
    }

    $pdo->beginTransaction();
    try {
        $stmt = $pdo->prepare('INSERT INTO proposals (type, title, status, created_by) VALUES (\'post\', :title, \'pending\', :created_by)');
        $stmt->execute([
            'title' => $title,
            'created_by' => $user['username'],
        ]);
        $proposalId = (int) $pdo->lastInsertId();

        $slideStmt = $pdo->prepare('INSERT INTO post_slides (proposal_id, position, format, text, text_align, image_path) VALUES (:proposal_id, :position, :format, :text, :text_align, :image_path)');

        $position = 1;
        foreach ($slides as $slide) {
            if (!is_array($slide)) {
                continue;
            }

            $format = normalize_format((string) ($slide['format'] ?? '1:1'));
            $textAlign = normalize_align((string) ($slide['text_align'] ?? 'left'));
            $text = trim((string) ($slide['text'] ?? ''));
            $imagePath = normalize_image_path((string) ($slide['image_path'] ?? ''));

            $slideStmt->execute([
                'proposal_id' => $proposalId,
                'position' => $position,
                'format' => $format,
                'text' => $text,
                'text_align' => $textAlign,
                'image_path' => $imagePath,
            ]);
            $position++;
        }

        $pdo->commit();
        echo json_encode(['ok' => true, 'id' => $proposalId]);
        exit;
    } catch (Throwable $error) {
        $pdo->rollBack();
        http_response_code(500);
        echo json_encode(['ok' => false, 'error' => 'db_insert']);
        exit;
    }
}

if ($action === 'create_reel') {
    if ($user['role'] !== 'creator') {
        http_response_code(403);
        echo json_encode(['ok' => false, 'error' => 'forbidden']);
        exit;
    }

    $title = trim((string) ($payload['title'] ?? ''));
    $scenes = $payload['scenes'] ?? [];

    if ($title === '' || !is_array($scenes) || count($scenes) === 0) {
        http_response_code(400);
        echo json_encode(['ok' => false, 'error' => 'missing_fields']);
        exit;
    }

    $pdo->beginTransaction();
    try {
        $stmt = $pdo->prepare('INSERT INTO proposals (type, title, status, created_by) VALUES (\'reel\', :title, \'pending\', :created_by)');
        $stmt->execute([
            'title' => $title,
            'created_by' => $user['username'],
        ]);
        $proposalId = (int) $pdo->lastInsertId();

        $sceneStmt = $pdo->prepare('INSERT INTO reel_scenes (proposal_id, position, duration_seconds, text_primary, text_secondary, background_path, background_opacity, overlay_path) VALUES (:proposal_id, :position, :duration_seconds, :text_primary, :text_secondary, :background_path, :background_opacity, :overlay_path)');

        $position = 1;
        foreach ($scenes as $scene) {
            if (!is_array($scene)) {
                continue;
            }

            $duration = (float) ($scene['duration'] ?? 3);
            if ($duration <= 0) {
                $duration = 3;
            }

            $sceneStmt->execute([
                'proposal_id' => $proposalId,
                'position' => $position,
                'duration_seconds' => $duration,
                'text_primary' => trim((string) ($scene['text_primary'] ?? '')),
                'text_secondary' => trim((string) ($scene['text_secondary'] ?? '')),
                'background_path' => normalize_image_path((string) ($scene['background_path'] ?? '')),
                'background_opacity' => normalize_opacity($scene['background_opacity'] ?? 0.2),
                'overlay_path' => normalize_image_path((string) ($scene['overlay_path'] ?? '')),
            ]);
            $position++;
        }

        $pdo->commit();
        echo json_encode(['ok' => true, 'id' => $proposalId]);
        exit;
    } catch (Throwable $error) {
        $pdo->rollBack();
        http_response_code(500);
        echo json_encode(['ok' => false, 'error' => 'db_insert']);
        exit;
    }
}

if ($action === 'update_status') {
    if ($user['role'] !== 'reviewer') {
        http_response_code(403);
        echo json_encode(['ok' => false, 'error' => 'forbidden']);
        exit;
    }

    $proposalId = (int) ($payload['id'] ?? 0);
    $status = (string) ($payload['status'] ?? '');
    if ($proposalId <= 0 || !in_array($status, ['approved', 'denied'], true)) {
        http_response_code(400);
        echo json_encode(['ok' => false, 'error' => 'invalid_data']);
        exit;
    }

    try {
        $stmt = $pdo->prepare('UPDATE proposals SET status = :status WHERE id = :id');
        $stmt->execute([
            'status' => $status,
            'id' => $proposalId,
        ]);
        echo json_encode(['ok' => true]);
        exit;
    } catch (Throwable $error) {
        http_response_code(500);
        echo json_encode(['ok' => false, 'error' => 'db_update']);
        exit;
    }
}

http_response_code(400);
echo json_encode(['ok' => false, 'error' => 'unknown_action']);

function load_proposals(): array
{
    try {
        $pdo = require __DIR__ . '/db.php';
    } catch (Throwable $error) {
        return [];
    }

    $stmt = $pdo->query('SELECT id, type, title, status, created_by, created_at, updated_at FROM proposals ORDER BY created_at DESC');
    $proposals = $stmt->fetchAll();
    if (!$proposals) {
        return [];
    }

    $ids = array_map(static fn ($row) => (int) $row['id'], $proposals);
    $placeholders = implode(',', array_fill(0, count($ids), '?'));

    $slides = [];
    $scenes = [];
    if (!empty($ids)) {
        $slideStmt = $pdo->prepare("SELECT proposal_id, position, format, text, text_align, image_path FROM post_slides WHERE proposal_id IN ($placeholders) ORDER BY position ASC");
        $slideStmt->execute($ids);
        foreach ($slideStmt->fetchAll() as $slide) {
            $slides[(int) $slide['proposal_id']][] = [
                'position' => (int) $slide['position'],
                'format' => (string) $slide['format'],
                'text' => (string) $slide['text'],
                'text_align' => (string) $slide['text_align'],
                'image_path' => (string) $slide['image_path'],
            ];
        }

        $sceneStmt = $pdo->prepare("SELECT proposal_id, position, duration_seconds, text_primary, text_secondary, background_path, background_opacity, overlay_path FROM reel_scenes WHERE proposal_id IN ($placeholders) ORDER BY position ASC");
        $sceneStmt->execute($ids);
        foreach ($sceneStmt->fetchAll() as $scene) {
            $scenes[(int) $scene['proposal_id']][] = [
                'position' => (int) $scene['position'],
                'duration' => (float) $scene['duration_seconds'],
                'text_primary' => (string) $scene['text_primary'],
                'text_secondary' => (string) $scene['text_secondary'],
                'background_path' => (string) $scene['background_path'],
                'background_opacity' => (float) $scene['background_opacity'],
                'overlay_path' => (string) $scene['overlay_path'],
            ];
        }
    }

    $result = [];
    foreach ($proposals as $proposal) {
        $id = (int) $proposal['id'];
        $entry = [
            'id' => $id,
            'type' => (string) $proposal['type'],
            'title' => (string) $proposal['title'],
            'status' => (string) $proposal['status'],
            'created_by' => (string) $proposal['created_by'],
            'created_at' => (string) $proposal['created_at'],
            'updated_at' => (string) $proposal['updated_at'],
        ];
        if ($entry['type'] === 'post') {
            $entry['slides'] = $slides[$id] ?? [];
        } else {
            $entry['scenes'] = $scenes[$id] ?? [];
        }
        $result[] = $entry;
    }

    return $result;
}

function normalize_format(string $format): string
{
    $allowed = ['1:1', '16:9', '9:16'];
    return in_array($format, $allowed, true) ? $format : '1:1';
}

function normalize_align(string $align): string
{
    $allowed = ['left', 'center', 'right'];
    return in_array($align, $allowed, true) ? $align : 'left';
}

function normalize_image_path(string $path): string
{
    $path = trim($path);
    if ($path === '') {
        return '';
    }

    $path = str_replace(['\\', '..'], ['/', ''], $path);
    return $path;
}

function normalize_opacity($value): float
{
    $opacity = (float) $value;
    if ($opacity < 0) {
        return 0;
    }
    if ($opacity > 1) {
        return 1;
    }
    return $opacity;
}
