const user = window.APP_USER || { role: "viewer" };

const state = {
  media: [],
  post: {
    title: "",
    slides: [],
    activeSlide: 0,
  },
  reel: {
    title: "",
    scenes: [],
  },
  proposals: [],
};

const elements = {
  builderPanel: document.getElementById("builderPanel"),
  tabs: document.querySelectorAll(".tab"),
  tabPanels: document.querySelectorAll("[data-tab-panel]"),
  slidesContainer: document.getElementById("slidesContainer"),
  scenesContainer: document.getElementById("scenesContainer"),
  postTitle: document.getElementById("postTitle"),
  reelTitle: document.getElementById("reelTitle"),
  addSlide: document.getElementById("addSlide"),
  addScene: document.getElementById("addScene"),
  savePost: document.getElementById("savePost"),
  saveReel: document.getElementById("saveReel"),
  postPreview: document.getElementById("postPreview"),
  reelPreview: document.getElementById("reelPreview"),
  pendingList: document.getElementById("pendingList"),
  approvedList: document.getElementById("approvedList"),
  deniedList: document.getElementById("deniedList"),
};

const formatOptions = [
  { value: "1:1", label: "1:1 (Cuadrado)" },
  { value: "16:9", label: "16:9 (Horizontal)" },
  { value: "9:16", label: "9:16 (Vertical)" },
];

const alignOptions = [
  { value: "left", label: "Izquierda" },
  { value: "center", label: "Centro" },
  { value: "right", label: "Derecha" },
];

let reelTimer = null;

const escapeHtml = (value) => {
  return String(value)
    .replace(/&/g, "&amp;")
    .replace(/</g, "&lt;")
    .replace(/>/g, "&gt;")
    .replace(/\"/g, "&quot;")
    .replace(/'/g, "&#039;");
};

const fitTextInBox = (element, maxSize = 64, minSize = 14) => {
  if (!element) return;
  const container = element.parentElement;
  if (!container) return;

  let low = minSize;
  let high = maxSize;
  let best = minSize;

  while (low <= high) {
    const mid = Math.floor((low + high) / 2);
    element.style.fontSize = `${mid}px`;
    if (element.scrollWidth <= container.clientWidth && element.scrollHeight <= container.clientHeight) {
      best = mid;
      low = mid + 1;
    } else {
      high = mid - 1;
    }
  }

  element.style.fontSize = `${best}px`;
};

const applyFitAll = (root) => {
  const elements = root.querySelectorAll(".frame-text.fit");
  elements.forEach((el) => fitTextInBox(el));
};

const ensureCreator = () => {
  if (user.role !== "creator") {
    elements.builderPanel.innerHTML = `
      <div class="panel-header">
        <h2>Solo Ignacio puede crear propuestas</h2>
        <p>Tu rol es revisor. Desde aquí puedes aprobar o denegar.</p>
      </div>
    `;
    return false;
  }
  return true;
};

const loadMedia = async () => {
  const response = await fetch("api/media.php", { cache: "no-store" });
  const data = await response.json();
  if (data.ok) {
    state.media = data.images || [];
  }
};

const loadProposals = async () => {
  const response = await fetch("api/proposals.php", { cache: "no-store" });
  const data = await response.json();
  if (data.ok) {
    state.proposals = data.data || [];
    renderBoard();
  }
};

const defaultSlide = () => ({
  text: "",
  format: "1:1",
  text_align: "left",
  image_path: state.media[0] || "",
});

const defaultScene = () => ({
  text_primary: "",
  text_secondary: "",
  duration: 3,
  background_path: "",
  background_opacity: 0.2,
  overlay_path: "",
});

const formatClass = (format) => `ratio-${format.replace(":", "-")}`;

const renderImagePicker = (selected, index, kind) => {
  const noneSelected = !selected;
  const noneThumb = `
    <div class="image-thumb ${noneSelected ? "selected" : ""}" data-kind="${kind}" data-index="${index}" data-path="">
      <span class="mini-text">Sin</span>
    </div>
  `;

  if (!state.media.length) {
    return `
      <div class="image-picker">
        ${noneThumb}
        <label class="upload-btn">
          Subir imagen
          <input type="file" data-upload-kind="${kind}" data-index="${index}" hidden />
        </label>
      </div>
    `;
  }

  const thumbs = state.media
    .map((path) => {
      const isSelected = path === selected;
      return `
        <div class="image-thumb ${isSelected ? "selected" : ""}" data-kind="${kind}" data-index="${index}" data-path="${escapeHtml(path)}">
          <img src="${escapeHtml(path)}" alt="imagen" />
        </div>
      `;
    })
    .join("");

  return `
    <div class="image-picker">
      ${noneThumb}
      ${thumbs}
      <label class="upload-btn">
        Subir imagen
        <input type="file" data-upload-kind="${kind}" data-index="${index}" hidden />
      </label>
    </div>
  `;
};

const renderSlides = () => {
  if (!elements.slidesContainer) return;

  elements.slidesContainer.innerHTML = state.post.slides
    .map((slide, index) => {
      const isActive = index === state.post.activeSlide ? "active" : "";
      const textClass = slide.text_align === "center" ? "full" : "half";
      return `
        <div class="slide-card ${isActive}" data-index="${index}">
          <div class="slide-header">
            <h4>Slide ${index + 1}</h4>
            <button class="btn ghost small" data-action="remove-slide" data-index="${index}" type="button">Eliminar</button>
          </div>
          <div class="slide-grid">
            <label class="field">
              <span>Formato</span>
              <select data-field="format" data-index="${index}">
                ${formatOptions
                  .map((opt) => `<option value="${opt.value}" ${opt.value === slide.format ? "selected" : ""}>${opt.label}</option>`)
                  .join("")}
              </select>
            </label>
            <label class="field">
              <span>Alineación texto</span>
              <select data-field="text_align" data-index="${index}">
                ${alignOptions
                  .map((opt) => `<option value="${opt.value}" ${opt.value === slide.text_align ? "selected" : ""}>${opt.label}</option>`)
                  .join("")}
              </select>
            </label>
          </div>
          <label class="field">
            <span>Texto</span>
            <textarea data-field="text" data-index="${index}" placeholder="Escribe aquí...">${escapeHtml(slide.text)}</textarea>
          </label>
          <div>
            <span class="field">Imagen</span>
            ${renderImagePicker(slide.image_path, index, "slide")}
          </div>
          <div class="preview-frame ${formatClass(slide.format)}">
            <div class="frame-inner">
              <div class="frame-text fit ${textClass} align-${slide.text_align}">${escapeHtml(slide.text || "Tu texto impactante")}</div>
              ${slide.image_path ? `<div class="frame-overlay"><img src="${escapeHtml(slide.image_path)}" alt="imagen" /></div>` : ""}
            </div>
          </div>
        </div>
      `;
    })
    .join("");

  applyFitAll(elements.slidesContainer);
  updatePostPreview();
};

const renderScenes = () => {
  if (!elements.scenesContainer) return;

  elements.scenesContainer.innerHTML = state.reel.scenes
    .map((scene, index) => {
      return `
        <div class="slide-card" data-index="${index}">
          <div class="slide-header">
            <h4>Escena ${index + 1}</h4>
            <button class="btn ghost small" data-action="remove-scene" data-index="${index}" type="button">Eliminar</button>
          </div>
          <div class="slide-grid">
            <label class="field">
              <span>Duración (s)</span>
              <input type="number" min="1" max="10" step="0.5" data-field="duration" data-index="${index}" value="${scene.duration}" />
            </label>
            <label class="field">
              <span>Opacidad fondo</span>
              <input type="number" min="0" max="1" step="0.1" data-field="background_opacity" data-index="${index}" value="${scene.background_opacity}" />
            </label>
          </div>
          <label class="field">
            <span>Texto principal</span>
            <textarea data-field="text_primary" data-index="${index}" placeholder="Hook principal">${escapeHtml(scene.text_primary)}</textarea>
          </label>
          <label class="field">
            <span>Texto secundario (opcional)</span>
            <textarea data-field="text_secondary" data-index="${index}" placeholder="Refuerzo">${escapeHtml(scene.text_secondary)}</textarea>
          </label>
          <div>
            <span class="field">Imagen de fondo</span>
            ${renderImagePicker(scene.background_path, index, "scene")}
          </div>
          <div>
            <span class="field">Imagen overlay (opcional)</span>
            ${renderImagePicker(scene.overlay_path, index, "overlay")}
          </div>
        </div>
      `;
    })
    .join("");

  updateReelPreview();
};

const updatePostPreview = () => {
  if (!elements.postPreview) return;
  const slide = state.post.slides[state.post.activeSlide] || state.post.slides[0];
  if (!slide) {
    elements.postPreview.innerHTML = "<p class=\"hint\">Añade un slide para ver el preview.</p>";
    return;
  }

  const textClass = slide.text_align === "center" ? "full" : "half";
  elements.postPreview.innerHTML = `
    <div class="preview-frame ${formatClass(slide.format)}">
      <div class="frame-inner">
        <div class="frame-text fit ${textClass} align-${slide.text_align}">${escapeHtml(slide.text || "Tu texto impactante")}</div>
        ${slide.image_path ? `<div class="frame-overlay"><img src="${escapeHtml(slide.image_path)}" alt="imagen" /></div>` : ""}
      </div>
    </div>
  `;

  applyFitAll(elements.postPreview);
};

const updateReelPreview = () => {
  if (!elements.reelPreview) return;
  if (reelTimer) {
    clearTimeout(reelTimer);
    reelTimer = null;
  }

  const scenes = state.reel.scenes;
  if (!scenes.length) {
    elements.reelPreview.innerHTML = "<p class=\"hint\">Añade escenas para ver el preview.</p>";
    return;
  }

  elements.reelPreview.innerHTML = `
    <div class="preview-frame ratio-9-16 reel">
      <div class="reel-player" id="reelPlayer"></div>
    </div>
  `;

  const player = document.getElementById("reelPlayer");
  if (!player) return;

  player.innerHTML = scenes
    .map((scene, idx) => {
      const bgStyle = scene.background_path ? `style=\"background-image:url('${escapeHtml(scene.background_path)}')\"` : "";
      return `
        <div class="reel-scene ${idx === 0 ? "active" : ""}" data-index="${idx}">
          <div class="frame-inner" ${bgStyle}>
            ${scene.background_path ? `<img class=\"bg\" src=\"${escapeHtml(scene.background_path)}\" alt=\"\" style=\"opacity:${scene.background_opacity}\" />` : ""}
            <div class="frame-text fit full align-center">${escapeHtml(scene.text_primary || "Texto principal")}</div>
            ${scene.text_secondary ? `<div class=\"frame-text full align-center\" style=\"margin-top: 90px; font-size: 18px; opacity: 0.8;\">${escapeHtml(scene.text_secondary)}</div>` : ""}
            ${scene.overlay_path ? `<div class=\"frame-overlay\" style=\"width: 55%; max-width: 180px;\"><img src=\"${escapeHtml(scene.overlay_path)}\" alt=\"overlay\" /></div>` : ""}
          </div>
        </div>
      `;
    })
    .join("");

  applyFitAll(player);

  const nodes = Array.from(player.querySelectorAll(".reel-scene"));
  let index = 0;

  const showScene = (idx) => {
    nodes.forEach((node) => node.classList.remove("active"));
    const current = nodes[idx];
    if (current) current.classList.add("active");
  };

  showScene(0);
  index = 1 % scenes.length;

  const cycle = () => {
    showScene(index);
    const duration = scenes[index]?.duration ? scenes[index].duration * 1000 : 3000;
    index = (index + 1) % scenes.length;
    reelTimer = setTimeout(cycle, duration);
  };

  reelTimer = setTimeout(cycle, scenes[0].duration * 1000 || 3000);
};

const renderBoard = () => {
  if (!elements.pendingList) return;

  elements.pendingList.innerHTML = "";
  elements.approvedList.innerHTML = "";
  elements.deniedList.innerHTML = "";

  state.proposals.forEach((proposal) => {
    const list = proposal.status === "approved" ? elements.approvedList : proposal.status === "denied" ? elements.deniedList : elements.pendingList;
    const badgeClass = proposal.type === "post" ? "post" : "reel";
    const preview = proposal.type === "post" ? proposal.slides?.[0] : proposal.scenes?.[0];
    const previewHtml = proposal.type === "post" ? renderMiniPost(preview) : renderMiniReel(preview);
    const actions =
      proposal.status === "pending" && user.role === "reviewer"
        ? `
          <div class="actions">
            <button class="btn ghost small" data-action="deny" data-id="${proposal.id}">Denegar</button>
            <button class="btn primary small" data-action="approve" data-id="${proposal.id}">Aprobar</button>
          </div>
        `
        : "";

    const card = document.createElement("div");
    card.className = "card-mini";
    card.innerHTML = `
      <div class="meta">
        <span class="badge ${badgeClass}">${proposal.type.toUpperCase()}</span>
        <span>${escapeHtml(proposal.created_by)}</span>
      </div>
      <div class="title">${escapeHtml(proposal.title)}</div>
      ${previewHtml}
      ${actions}
    `;

    list.appendChild(card);
  });
};

const renderMiniPost = (slide) => {
  if (!slide) {
    return `<div class="mini-preview"><div class="mini-text">Sin preview</div></div>`;
  }
  const text = slide.text ? slide.text.slice(0, 40) : "Post";
  return `
    <div class="mini-preview">
      ${slide.image_path ? `<img src="${escapeHtml(slide.image_path)}" alt="" />` : ""}
      <div class="mini-text">${escapeHtml(text)}</div>
    </div>
  `;
};

const renderMiniReel = (scene) => {
  if (!scene) {
    return `<div class="mini-preview"><div class="mini-text">Sin preview</div></div>`;
  }
  const text = scene.text_primary ? scene.text_primary.slice(0, 40) : "Reel";
  return `
    <div class="mini-preview">
      ${scene.background_path ? `<img src="${escapeHtml(scene.background_path)}" alt="" style="opacity: 0.35;" />` : ""}
      <div class="mini-text">${escapeHtml(text)}</div>
    </div>
  `;
};

const setTab = (tabName) => {
  elements.tabs.forEach((tab) => {
    tab.classList.toggle("active", tab.dataset.tab === tabName);
  });
  elements.tabPanels.forEach((panel) => {
    panel.classList.toggle("active", panel.dataset.tabPanel === tabName);
  });
};

const initTabs = () => {
  elements.tabs.forEach((tab) => {
    tab.addEventListener("click", () => setTab(tab.dataset.tab));
  });
};

const uploadImage = async (file) => {
  const formData = new FormData();
  formData.append("file", file);
  const response = await fetch("api/upload.php", {
    method: "POST",
    body: formData,
  });
  const data = await response.json();
  if (data.ok) {
    if (!state.media.includes(data.path)) {
      state.media.unshift(data.path);
    }
    return data.path;
  }
  throw new Error(data.error || "upload_failed");
};

const initPostForm = () => {
  if (!elements.postTitle) return;
  state.post.slides = [defaultSlide()];
  state.post.activeSlide = 0;
  elements.postTitle.addEventListener("input", (event) => {
    state.post.title = event.target.value;
  });

  elements.addSlide.addEventListener("click", () => {
    state.post.slides.push(defaultSlide());
    state.post.activeSlide = state.post.slides.length - 1;
    renderSlides();
  });

  elements.savePost.addEventListener("click", async () => {
    const title = elements.postTitle.value.trim();
    if (!title) {
      alert("Añade un título.");
      return;
    }
    if (!state.post.slides.length) {
      alert("Añade al menos un slide.");
      return;
    }

    const payload = {
      action: "create_post",
      title,
      slides: state.post.slides,
    };

    const response = await fetch("api/proposals.php", {
      method: "POST",
      headers: { "Content-Type": "application/json" },
      body: JSON.stringify(payload),
    });
    const data = await response.json();
    if (data.ok) {
      elements.postTitle.value = "";
      state.post = { title: "", slides: [defaultSlide()], activeSlide: 0 };
      renderSlides();
      await loadProposals();
      alert("Propuesta de post guardada.");
    } else {
      alert("No se pudo guardar.");
    }
  });

  elements.slidesContainer.addEventListener("input", (event) => {
    const target = event.target;
    const index = Number(target.dataset.index);
    if (Number.isNaN(index)) return;
    const field = target.dataset.field;
    if (!field) return;
    state.post.slides[index][field] = target.value;
    state.post.activeSlide = index;
    updatePostPreview();
    applyFitAll(elements.slidesContainer);
  });

  elements.slidesContainer.addEventListener("change", async (event) => {
    const target = event.target;
    if (target.matches("input[type='file']")) {
      const index = Number(target.dataset.index);
      const kind = target.dataset.uploadKind;
      const file = target.files?.[0];
      if (!file) return;
      try {
        const path = await uploadImage(file);
        if (kind === "slide") {
          state.post.slides[index].image_path = path;
        }
        renderSlides();
      } catch (error) {
        alert("Error subiendo la imagen.");
      }
    }
  });

  elements.slidesContainer.addEventListener("click", (event) => {
    const thumb = event.target.closest(".image-thumb");
    if (thumb) {
      const index = Number(thumb.dataset.index);
      const kind = thumb.dataset.kind;
      const path = thumb.dataset.path;
      if (kind === "slide") {
        state.post.slides[index].image_path = path;
        renderSlides();
      }
      return;
    }

    const removeBtn = event.target.closest("[data-action='remove-slide']");
    if (removeBtn) {
      const index = Number(removeBtn.dataset.index);
      state.post.slides.splice(index, 1);
      state.post.activeSlide = Math.max(0, state.post.activeSlide - 1);
      if (!state.post.slides.length) {
        state.post.slides.push(defaultSlide());
      }
      renderSlides();
    }
  });

  renderSlides();
};

const initReelForm = () => {
  if (!elements.reelTitle) return;
  state.reel.scenes = [defaultScene()];

  elements.reelTitle.addEventListener("input", (event) => {
    state.reel.title = event.target.value;
  });

  elements.addScene.addEventListener("click", () => {
    state.reel.scenes.push(defaultScene());
    renderScenes();
  });

  elements.saveReel.addEventListener("click", async () => {
    const title = elements.reelTitle.value.trim();
    if (!title) {
      alert("Añade un título.");
      return;
    }
    if (!state.reel.scenes.length) {
      alert("Añade al menos una escena.");
      return;
    }

    const payload = {
      action: "create_reel",
      title,
      scenes: state.reel.scenes,
    };

    const response = await fetch("api/proposals.php", {
      method: "POST",
      headers: { "Content-Type": "application/json" },
      body: JSON.stringify(payload),
    });
    const data = await response.json();
    if (data.ok) {
      elements.reelTitle.value = "";
      state.reel = { title: "", scenes: [defaultScene()] };
      renderScenes();
      await loadProposals();
      alert("Propuesta de reel guardada.");
    } else {
      alert("No se pudo guardar.");
    }
  });

  elements.scenesContainer.addEventListener("input", (event) => {
    const target = event.target;
    const index = Number(target.dataset.index);
    if (Number.isNaN(index)) return;
    const field = target.dataset.field;
    if (!field) return;
    if (field === "duration" || field === "background_opacity") {
      state.reel.scenes[index][field] = Number(target.value);
    } else {
      state.reel.scenes[index][field] = target.value;
    }
    updateReelPreview();
  });

  elements.scenesContainer.addEventListener("change", async (event) => {
    const target = event.target;
    if (target.matches("input[type='file']")) {
      const index = Number(target.dataset.index);
      const kind = target.dataset.uploadKind;
      const file = target.files?.[0];
      if (!file) return;
      try {
        const path = await uploadImage(file);
        if (kind === "scene") {
          state.reel.scenes[index].background_path = path;
        }
        if (kind === "overlay") {
          state.reel.scenes[index].overlay_path = path;
        }
        renderScenes();
      } catch (error) {
        alert("Error subiendo la imagen.");
      }
    }
  });

  elements.scenesContainer.addEventListener("click", (event) => {
    const thumb = event.target.closest(".image-thumb");
    if (thumb) {
      const index = Number(thumb.dataset.index);
      const kind = thumb.dataset.kind;
      const path = thumb.dataset.path;
      if (kind === "scene") {
        state.reel.scenes[index].background_path = path;
      }
      if (kind === "overlay") {
        state.reel.scenes[index].overlay_path = path;
      }
      renderScenes();
      return;
    }

    const removeBtn = event.target.closest("[data-action='remove-scene']");
    if (removeBtn) {
      const index = Number(removeBtn.dataset.index);
      state.reel.scenes.splice(index, 1);
      if (!state.reel.scenes.length) {
        state.reel.scenes.push(defaultScene());
      }
      renderScenes();
    }
  });

  renderScenes();
};

const initBoardActions = () => {
  document.body.addEventListener("click", async (event) => {
    const approveBtn = event.target.closest("[data-action='approve']");
    const denyBtn = event.target.closest("[data-action='deny']");

    if (approveBtn || denyBtn) {
      const id = Number((approveBtn || denyBtn).dataset.id);
      const status = approveBtn ? "approved" : "denied";
      const response = await fetch("api/proposals.php", {
        method: "POST",
        headers: { "Content-Type": "application/json" },
        body: JSON.stringify({ action: "update_status", id, status }),
      });
      const data = await response.json();
      if (data.ok) {
        await loadProposals();
      } else {
        alert("No se pudo actualizar el estado.");
      }
    }
  });
};

const initApp = async () => {
  if (!user || !user.username) return;

  await loadMedia();
  await loadProposals();

  if (ensureCreator()) {
    initTabs();
    initPostForm();
    initReelForm();
  }

  initBoardActions();
};

initApp();
